<?php
/**
 * Image helper class
 *
 * @package		CSVIVirtueMart
 * @author 		Roland Dalmulder
 * @link 		http://www.csvimproved.com
 * @copyright 	Copyright (C) 2006 - 2011 RolandD Cyber Produksi
 * @version 	$Id: images.php 1623 2011-09-28 17:39:57Z RolandD $
 */
defined( '_JEXEC' ) or die( 'Direct Access to this location is not allowed.' );

/**
 * Image converter
 *
 * @package CSVIVirtueMart
 */
class ImageHelper {

	/** @var array holds the mime types */
	private $_mime_types = array();
	/** @var array holds the image types */
	private $_image_types = array();
	/** @var array holds the mime types it found */
	private $_found_mime_type = array();
	/** @var array contains all the image data for processing */
	private $_imagedata = array();
	/** @var array all supported image extensions */
	private $_supported_images = array('jpg', 'jpeg', 'gif', 'png', 'bmp');

	/**	@var int $bg_red 0-255 - red color variable for background filler */
	private $bg_red = 0;
	/**	@var int $bg_green 0-255 - green color variable for background filler */
	private $bg_green = 0;
	/** @var int $bg_blue 0-255 - blue color variable for background filler */
	private $bg_blue = 0;
	/**	@var int $maxSize 0-1 - true/false - should thumbnail be filled to max pixels */
	private $maxSize = false;
	/** @var string $file the original file */
	private $file = null;
	/** @var string $file_extension the extension of the original file */
	private $file_extension = null;
	/** @var string $file_out the name of the file to be created */
	private $file_out = null;
	/** @var string $file_out_extension the extension of the file to be created */
	public $file_out_extension = null;
	/** @var int $file_out_width the width of the file to be generated */
	private $file_out_width = 0;
	/** @var int $file_out_height the height of the file to be generated */
	private $file_out_height = 0;
	/** @var bool $rename set if the file should be renamed including the size of the image */
	private $rename = 0;

	/**
	* Construct the class and load some basic settings
	*
	* @author RolandD
	* @access public
	*/
	public function __construct() {
		$this->_loadMimeTypes();
		$this->_loadImageTypes();
	}

	/**
	 * Check if the given file is an image
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo		add support for external files
	 * @see
	 * @access 		public
	 * @param 		string	$file	full path to file to check
	 * @param 		bool	$remote	true if the file to check is a remote file
	 * @return 		bool	true if file is image, false if file is not an image
	 * @since		3.0
	 */
	public function isImage($file, $remote=false) {
		$mime_type = $this->findMimeType($file, $remote);
		if ($mime_type) {
			foreach ($this->_image_types as $key => $type) {
				if ($type['mime_type'] == $mime_type) return true;
			}
		}
		// If we get here, no image type has been found
		return false;
	}

	/**
	 * Check a file for its mime type
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		public
	 * @param		string 	$filename	the full location of the file to check
	 * @param 		bool	$remote 	true if the file to check is a remote file
	 * @return 		string|bool 	return mime type if found | return false if no type found
	 * @since 		3.0
	 */
	public function findMimeType($filename, $remote=false) {
		jimport('joomla.filesystem.file');
		if (JFile::exists($filename) || $remote) {
			$handle = @fopen($filename, "r");
			if ($handle) {
				$string = fread($handle, 20);
				$max_length_found = 0;
				foreach ($this->_mime_types as $key => $type) {
					if (stripos(bin2hex($string), $type['signature'], 0) !== false) {
						if (strlen($type['signature']) > $max_length_found) {
							$max_length_found = strlen($type['signature']);
							$this->_found_mime_type['mime_type'] = $type['mime_type'];
						}
					}
				}
				fclose($handle);
				if (isset($this->_found_mime_type['mime_type'])) return $this->_found_mime_type['mime_type'];
				else return false;
			}
			else {
				// Cannot open the image file, do a simple check
				switch (JFile::getExt($filename)) {
					case 'jpg':
					case 'jpeg':
						return 'image/jpeg';
						break;
					case 'png':
						return 'image/png';
						break;
					case 'gif':
						return 'image/gif';
						break;
					case 'bmp':
						return 'image/bmp';
						break;
					default:
						return false;
						break;
				}
			}
		}
		else return false;
	}

	/**
	 * Load mime type signatures
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return 		void
	 * @since 		3.0
	 */
	private function _loadMimeTypes() {
		$this->_mime_types[0]['signature'] = '474946383761';
		$this->_mime_types[1]['signature'] = '424D';
		$this->_mime_types[2]['signature'] = '4D5A';
		$this->_mime_types[3]['signature'] = '504B0304';
		$this->_mime_types[4]['signature'] = 'D0CF11E0A1B11AE1';
		$this->_mime_types[5]['signature'] = '0100000058000000';
		$this->_mime_types[6]['signature'] = '03000000C466C456';
		$this->_mime_types[7]['signature'] = '3F5F0300';
		$this->_mime_types[8]['signature'] = '1F8B08';
		$this->_mime_types[9]['signature'] = '28546869732066696C65';
		$this->_mime_types[10]['signature'] = '0000010000';
		$this->_mime_types[11]['signature'] = '4C000000011402';
		$this->_mime_types[12]['signature'] = '25504446';
		$this->_mime_types[13]['signature'] = '5245474544495434';
		$this->_mime_types[14]['signature'] = '7B5C727466';
		$this->_mime_types[15]['signature'] = 'lh';
		$this->_mime_types[16]['signature'] = 'MThd';
		$this->_mime_types[17]['signature'] = '0A050108';
		$this->_mime_types[18]['signature'] = '25215053';
		$this->_mime_types[19]['signature'] = '2112';
		$this->_mime_types[20]['signature'] = '1A02';
		$this->_mime_types[21]['signature'] = '1A03';
		$this->_mime_types[22]['signature'] = '1A04';
		$this->_mime_types[23]['signature'] = '1A08';
		$this->_mime_types[24]['signature'] = '1A09';
		$this->_mime_types[25]['signature'] = '60EA';
		$this->_mime_types[26]['signature'] = '41564920';
		$this->_mime_types[27]['signature'] = '425A68';
		$this->_mime_types[28]['signature'] = '49536328';
		$this->_mime_types[29]['signature'] = '4C01';
		$this->_mime_types[30]['signature'] = '303730373037';
		$this->_mime_types[31]['signature'] = '4352555348';
		$this->_mime_types[32]['signature'] = '3ADE68B1';
		$this->_mime_types[33]['signature'] = '1F8B';
		$this->_mime_types[34]['signature'] = '91334846';
		$this->_mime_types[35]['signature'] = '3C68746D6C3E';
		$this->_mime_types[36]['signature'] = '3C48544D4C3E';
		$this->_mime_types[37]['signature'] = '3C21444F4354';
		$this->_mime_types[38]['signature'] = '100';
		$this->_mime_types[39]['signature'] = '5F27A889';
		$this->_mime_types[40]['signature'] = '2D6C68352D';
		$this->_mime_types[41]['signature'] = '20006040600';
		$this->_mime_types[42]['signature'] = '00001A0007800100';
		$this->_mime_types[43]['signature'] = '00001A0000100400';
		$this->_mime_types[44]['signature'] = '20006800200';
		$this->_mime_types[45]['signature'] = '00001A0002100400';
		$this->_mime_types[46]['signature'] = '5B7665725D';
		$this->_mime_types[47]['signature'] = '300000041505052';
		$this->_mime_types[48]['signature'] = '1A0000030000';
		$this->_mime_types[49]['signature'] = '4D47582069747064';
		$this->_mime_types[50]['signature'] = '4D534346';
		$this->_mime_types[51]['signature'] = '4D546864';
		$this->_mime_types[52]['signature'] = '000001B3';
		$this->_mime_types[53]['signature'] = '0902060000001000B9045C00';
		$this->_mime_types[54]['signature'] = '0904060000001000F6055C00';
		$this->_mime_types[55]['signature'] = '7FFE340A';
		$this->_mime_types[56]['signature'] = '1234567890FF';
		$this->_mime_types[57]['signature'] = '31BE000000AB0000';
		$this->_mime_types[58]['signature'] = '1A00000300001100';
		$this->_mime_types[59]['signature'] = '7E424B00';
		$this->_mime_types[60]['signature'] = '504B0304';
		$this->_mime_types[61]['signature'] = '89504E470D0A';
		$this->_mime_types[62]['signature'] = '6D646174';
		$this->_mime_types[63]['signature'] = '6D646174';
		$this->_mime_types[64]['signature'] = '52617221';
		$this->_mime_types[65]['signature'] = '2E7261FD';
		$this->_mime_types[66]['signature'] = 'EDABEEDB';
		$this->_mime_types[67]['signature'] = '2E736E64';
		$this->_mime_types[68]['signature'] = '53495421';
		$this->_mime_types[69]['signature'] = '53747566664974';
		$this->_mime_types[70]['signature'] = '1F9D';
		$this->_mime_types[71]['signature'] = '49492A';
		$this->_mime_types[72]['signature'] = '4D4D2A';
		$this->_mime_types[73]['signature'] = '554641';
		$this->_mime_types[74]['signature'] = '57415645666D74';
		$this->_mime_types[75]['signature'] = 'D7CDC69A';
		$this->_mime_types[76]['signature'] = '4C000000';
		$this->_mime_types[77]['signature'] = '504B3030504B0304';
		$this->_mime_types[78]['signature'] = 'FF575047';
		$this->_mime_types[79]['signature'] = 'FF575043';
		$this->_mime_types[80]['signature'] = '3C3F786D6C';
		$this->_mime_types[81]['signature'] = 'FFFE3C0052004F004F0054005300540055004200';
		$this->_mime_types[82]['signature'] = '3C21454E54495459';
		$this->_mime_types[83]['signature'] = '5A4F4F20';
		$this->_mime_types[84]['signature'] = 'FFD8FFFE';
		$this->_mime_types[85]['signature'] = 'FFD8FFE0';
		$this->_mime_types[86]['signature'] = 'FFD8FFE1';
		$this->_mime_types[87]['signature'] = 'FFD8FFDB';
		$this->_mime_types[88]['signature'] = '474946383961';

		// Extensions
		$this->_mime_types[0]['extension'] = '.gif';
		$this->_mime_types[1]['extension'] = '.bmp';
		$this->_mime_types[2]['extension'] = '.exe;.com;.386;.ax;.acm;.sys;.dll;.drv;.flt;.fon;.ocx;.scr;.lrc;.vxd;.cpl;.x32';
		$this->_mime_types[3]['extension'] = '.zip';
		$this->_mime_types[4]['extension'] = '.doc;.xls;.xlt;.ppt;.apr';
		$this->_mime_types[5]['extension'] = '.emf';
		$this->_mime_types[6]['extension'] = '.evt';
		$this->_mime_types[7]['extension'] = '.gid;.hlp;.lhp';
		$this->_mime_types[8]['extension'] = '.gz';
		$this->_mime_types[9]['extension'] = '.hqx';
		$this->_mime_types[10]['extension'] = '.ico';
		$this->_mime_types[11]['extension'] = '.lnk';
		$this->_mime_types[12]['extension'] = '.pdf';
		$this->_mime_types[13]['extension'] = '.reg';
		$this->_mime_types[14]['extension'] = '.rtf';
		$this->_mime_types[15]['extension'] = '.lzh';
		$this->_mime_types[16]['extension'] = '.mid';
		$this->_mime_types[17]['extension'] = '.pcx';
		$this->_mime_types[18]['extension'] = '.eps';
		$this->_mime_types[19]['extension'] = '.ain';
		$this->_mime_types[20]['extension'] = '.arc';
		$this->_mime_types[21]['extension'] = '.arc';
		$this->_mime_types[22]['extension'] = '.arc';
		$this->_mime_types[23]['extension'] = '.arc';
		$this->_mime_types[24]['extension'] = '.arc';
		$this->_mime_types[25]['extension'] = '.arj';
		$this->_mime_types[26]['extension'] = '.avi';
		$this->_mime_types[27]['extension'] = '.bz;.bz2';
		$this->_mime_types[28]['extension'] = '.cab';
		$this->_mime_types[29]['extension'] = '.obj';
		$this->_mime_types[30]['extension'] = '.tar;.cpio';
		$this->_mime_types[31]['extension'] = '.cru;.crush';
		$this->_mime_types[32]['extension'] = '.dcx';
		$this->_mime_types[33]['extension'] = '.gz;.tar;.tgz';
		$this->_mime_types[34]['extension'] = '.hap';
		$this->_mime_types[35]['extension'] = '.htm;.html';
		$this->_mime_types[36]['extension'] = '.htm;.html';
		$this->_mime_types[37]['extension'] = '.htm;.html';
		$this->_mime_types[38]['extension'] = '.ico';
		$this->_mime_types[39]['extension'] = '.jar';
		$this->_mime_types[40]['extension'] = '.lha';
		$this->_mime_types[41]['extension'] = '.wk1;.wks';
		$this->_mime_types[42]['extension'] = '.fm3';
		$this->_mime_types[43]['extension'] = '.wk3';
		$this->_mime_types[44]['extension'] = '.fmt';
		$this->_mime_types[45]['extension'] = '.wk4';
		$this->_mime_types[46]['extension'] = '.ami';
		$this->_mime_types[47]['extension'] = '.adx';
		$this->_mime_types[48]['extension'] = '.nsf;.ntf';
		$this->_mime_types[49]['extension'] = '.ds4';
		$this->_mime_types[50]['extension'] = '.cab';
		$this->_mime_types[51]['extension'] = '.mid';
		$this->_mime_types[52]['extension'] = '.mpg;.mpeg';
		$this->_mime_types[53]['extension'] = '.xls';
		$this->_mime_types[54]['extension'] = '.xls';
		$this->_mime_types[55]['extension'] = '.doc';
		$this->_mime_types[56]['extension'] = '.doc';
		$this->_mime_types[57]['extension'] = '.doc';
		$this->_mime_types[58]['extension'] = '.nsf';
		$this->_mime_types[59]['extension'] = '.psp';
		$this->_mime_types[60]['extension'] = '.zip';
		$this->_mime_types[61]['extension'] = '.png';
		$this->_mime_types[62]['extension'] = '.mov';
		$this->_mime_types[63]['extension'] = '.qt';
		$this->_mime_types[64]['extension'] = '.rar';
		$this->_mime_types[65]['extension'] = '.ra;.ram';
		$this->_mime_types[66]['extension'] = '.rpm';
		$this->_mime_types[67]['extension'] = '.au';
		$this->_mime_types[68]['extension'] = '.sit';
		$this->_mime_types[69]['extension'] = '.sit';
		$this->_mime_types[70]['extension'] = '.z';
		$this->_mime_types[71]['extension'] = '.tif;.tiff';
		$this->_mime_types[72]['extension'] = '.tif;.tiff';
		$this->_mime_types[73]['extension'] = '.ufa';
		$this->_mime_types[74]['extension'] = '.wav';
		$this->_mime_types[75]['extension'] = '.wmf';
		$this->_mime_types[76]['extension'] = '.lnk';
		$this->_mime_types[77]['extension'] = '.zip';
		$this->_mime_types[78]['extension'] = '.wpg';
		$this->_mime_types[79]['extension'] = '.wp';
		$this->_mime_types[80]['extension'] = '.xml';
		$this->_mime_types[81]['extension'] = '.xml';
		$this->_mime_types[82]['extension'] = '.dtd';
		$this->_mime_types[83]['extension'] = '.zoo';
		$this->_mime_types[84]['extension'] = '.jpeg;.jpe;.jpg';
		$this->_mime_types[85]['extension'] = '.jpeg;.jpe;.jpg';
		$this->_mime_types[86]['extension'] = '.jpeg;.jpe;.jpg';
		$this->_mime_types[87]['extension'] = '.jpeg;.jpe;.jpg';
		$this->_mime_types[88]['extension'] = '.gif';

		// Descriptions
		$this->_mime_types[0]['description'] = 'GIF 87A';
		$this->_mime_types[1]['description'] = 'Windows Bitmap';
		$this->_mime_types[2]['description'] = 'Executable File ';
		$this->_mime_types[3]['description'] = 'Zip Compressed';
		$this->_mime_types[4]['description'] = 'MS Compound Document v1 or Lotus Approach APR file';
		$this->_mime_types[5]['description'] = 'xtended (Enhanced) Windows Metafile Format';
		$this->_mime_types[6]['description'] = 'Windows NT/2000 Event Viewer Log File';
		$this->_mime_types[7]['description'] = 'Windows Help File';
		$this->_mime_types[8]['description'] = 'GZ Compressed File';
		$this->_mime_types[9]['description'] = 'Macintosh BinHex 4 Compressed Archive';
		$this->_mime_types[10]['description'] = 'Icon File';
		$this->_mime_types[11]['description'] = 'Windows Link File';
		$this->_mime_types[12]['description'] = 'Adobe PDF File';
		$this->_mime_types[13]['description'] = 'Registry Data File';
		$this->_mime_types[14]['description'] = 'Rich Text Format File';
		$this->_mime_types[15]['description'] = 'Lzh compression file';
		$this->_mime_types[16]['description'] = 'Musical Instrument Digital Interface MIDI-sequention Sound';
		$this->_mime_types[17]['description'] = 'PC Paintbrush Bitmap Graphic';
		$this->_mime_types[18]['description'] = 'Adobe EPS File';
		$this->_mime_types[19]['description'] = 'AIN Archive File';
		$this->_mime_types[20]['description'] = 'ARC/PKPAK Compressed 1';
		$this->_mime_types[21]['description'] = 'ARC/PKPAK Compressed 2';
		$this->_mime_types[22]['description'] = 'ARC/PKPAK Compressed 3';
		$this->_mime_types[23]['description'] = 'ARC/PKPAK Compressed 4';
		$this->_mime_types[24]['description'] = 'ARC/PKPAK Compressed 5';
		$this->_mime_types[25]['description'] = 'ARJ Compressed';
		$this->_mime_types[26]['description'] = 'Audio Video Interleave (AVI)';
		$this->_mime_types[27]['description'] = 'Bzip Archive';
		$this->_mime_types[28]['description'] = 'Cabinet File';
		$this->_mime_types[29]['description'] = 'Compiled Object Module';
		$this->_mime_types[30]['description'] = 'CPIO Archive File';
		$this->_mime_types[31]['description'] = 'CRUSH Archive File';
		$this->_mime_types[32]['description'] = 'DCX Graphic File';
		$this->_mime_types[33]['description'] = 'Gzip Archive File';
		$this->_mime_types[34]['description'] = 'HAP Archive File';
		$this->_mime_types[35]['description'] = 'HyperText Markup Language 1';
		$this->_mime_types[36]['description'] = 'HyperText Markup Language 2';
		$this->_mime_types[37]['description'] = 'HyperText Markup Language 3';
		$this->_mime_types[38]['description'] = 'ICON File';
		$this->_mime_types[39]['description'] = 'JAR Archive File';
		$this->_mime_types[40]['description'] = 'LHA Compressed';
		$this->_mime_types[41]['description'] = 'Lotus 123 v1 Worksheet';
		$this->_mime_types[42]['description'] = 'Lotus 123 v3 FMT file';
		$this->_mime_types[43]['description'] = 'Lotus 123 v3 Worksheet';
		$this->_mime_types[44]['description'] = 'Lotus 123 v4 FMT file';
		$this->_mime_types[45]['description'] = 'Lotus 123 v5';
		$this->_mime_types[46]['description'] = 'Lotus Ami Pro';
		$this->_mime_types[47]['description'] = 'Lotus Approach ADX file';
		$this->_mime_types[48]['description'] = 'Lotus Notes Database/Template';
		$this->_mime_types[49]['description'] = 'Micrografix Designer 4';
		$this->_mime_types[50]['description'] = 'Microsoft CAB File Format';
		$this->_mime_types[51]['description'] = 'Midi Audio File';
		$this->_mime_types[52]['description'] = 'MPEG Movie';
		$this->_mime_types[53]['description'] = 'MS Excel v2';
		$this->_mime_types[54]['description'] = 'MS Excel v4';
		$this->_mime_types[55]['description'] = 'MS Word';
		$this->_mime_types[56]['description'] = 'MS Word 6.0';
		$this->_mime_types[57]['description'] = 'MS Word for DOS 6.0';
		$this->_mime_types[58]['description'] = 'Notes Database';
		$this->_mime_types[59]['description'] = 'PaintShop Pro Image File';
		$this->_mime_types[60]['description'] = 'PKZIP Compressed';
		$this->_mime_types[61]['description'] = 'PNG Image File';
		$this->_mime_types[62]['description'] = 'QuickTime Movie';
		$this->_mime_types[63]['description'] = 'Quicktime Movie File';
		$this->_mime_types[64]['description'] = 'RAR Archive File';
		$this->_mime_types[65]['description'] = 'Real Audio File';
		$this->_mime_types[66]['description'] = 'RPM Archive File';
		$this->_mime_types[67]['description'] = 'SoundMachine Audio File';
		$this->_mime_types[68]['description'] = 'Stuffit v1 Archive File';
		$this->_mime_types[69]['description'] = 'Stuffit v5 Archive File';
		$this->_mime_types[70]['description'] = 'TAR Compressed Archive File';
		$this->_mime_types[71]['description'] = 'TIFF (Intel)';
		$this->_mime_types[72]['description'] = 'TIFF (Motorola)';
		$this->_mime_types[73]['description'] = 'UFA Archive File';
		$this->_mime_types[74]['description'] = 'Wave Files';
		$this->_mime_types[75]['description'] = 'Windows Meta File';
		$this->_mime_types[76]['description'] = 'Windows Shortcut (Link File)';
		$this->_mime_types[77]['description'] = 'WINZIP Compressed';
		$this->_mime_types[78]['description'] = 'WordPerfect Graphics';
		$this->_mime_types[79]['description'] = 'WordPerfect v5 or v6';
		$this->_mime_types[80]['description'] = 'XML Document';
		$this->_mime_types[81]['description'] = 'XML Document (ROOTSTUB)';
		$this->_mime_types[82]['description'] = 'XML DTD';
		$this->_mime_types[83]['description'] = 'ZOO Archive File';
		$this->_mime_types[84]['description'] = 'JPG Graphic File';
		$this->_mime_types[85]['description'] = 'JPG Graphic File';
		$this->_mime_types[86]['description'] = 'JPG Graphic File';
		$this->_mime_types[87]['description'] = 'JPG Graphic File';
		$this->_mime_types[88]['description'] = 'GIF 89A';

		// Mime descriptions
		$this->_mime_types[0]['mime_type'] = 'image/gif';
		$this->_mime_types[1]['mime_type'] = 'image/bmp';
		$this->_mime_types[2]['mime_type'] = '';
		$this->_mime_types[3]['mime_type'] = '';
		$this->_mime_types[4]['mime_type'] = '';
		$this->_mime_types[5]['mime_type'] = '';
		$this->_mime_types[6]['mime_type'] = '';
		$this->_mime_types[7]['mime_type'] = '';
		$this->_mime_types[8]['mime_type'] = '';
		$this->_mime_types[9]['mime_type'] = '';
		$this->_mime_types[10]['mime_type'] = '';
		$this->_mime_types[11]['mime_type'] = '';
		$this->_mime_types[12]['mime_type'] = 'application/pdf';
		$this->_mime_types[13]['mime_type'] = '';
		$this->_mime_types[14]['mime_type'] = '';
		$this->_mime_types[15]['mime_type'] = '';
		$this->_mime_types[16]['mime_type'] = '';
		$this->_mime_types[17]['mime_type'] = '';
		$this->_mime_types[18]['mime_type'] = '';
		$this->_mime_types[19]['mime_type'] = '';
		$this->_mime_types[20]['mime_type'] = '';
		$this->_mime_types[21]['mime_type'] = '';
		$this->_mime_types[22]['mime_type'] = '';
		$this->_mime_types[23]['mime_type'] = '';
		$this->_mime_types[24]['mime_type'] = '';
		$this->_mime_types[25]['mime_type'] = '';
		$this->_mime_types[26]['mime_type'] = '';
		$this->_mime_types[27]['mime_type'] = '';
		$this->_mime_types[28]['mime_type'] = '';
		$this->_mime_types[29]['mime_type'] = '';
		$this->_mime_types[30]['mime_type'] = '';
		$this->_mime_types[31]['mime_type'] = '';
		$this->_mime_types[32]['mime_type'] = '';
		$this->_mime_types[33]['mime_type'] = '';
		$this->_mime_types[34]['mime_type'] = '';
		$this->_mime_types[35]['mime_type'] = '';
		$this->_mime_types[36]['mime_type'] = '';
		$this->_mime_types[37]['mime_type'] = '';
		$this->_mime_types[38]['mime_type'] = '';
		$this->_mime_types[39]['mime_type'] = '';
		$this->_mime_types[40]['mime_type'] = '';
		$this->_mime_types[41]['mime_type'] = '';
		$this->_mime_types[42]['mime_type'] = '';
		$this->_mime_types[43]['mime_type'] = '';
		$this->_mime_types[44]['mime_type'] = '';
		$this->_mime_types[45]['mime_type'] = '';
		$this->_mime_types[46]['mime_type'] = '';
		$this->_mime_types[47]['mime_type'] = '';
		$this->_mime_types[48]['mime_type'] = '';
		$this->_mime_types[49]['mime_type'] = '';
		$this->_mime_types[50]['mime_type'] = '';
		$this->_mime_types[51]['mime_type'] = '';
		$this->_mime_types[52]['mime_type'] = '';
		$this->_mime_types[53]['mime_type'] = '';
		$this->_mime_types[54]['mime_type'] = '';
		$this->_mime_types[55]['mime_type'] = '';
		$this->_mime_types[56]['mime_type'] = '';
		$this->_mime_types[57]['mime_type'] = '';
		$this->_mime_types[58]['mime_type'] = '';
		$this->_mime_types[59]['mime_type'] = '';
		$this->_mime_types[60]['mime_type'] = '';
		$this->_mime_types[61]['mime_type'] = 'image/png';
		$this->_mime_types[62]['mime_type'] = '';
		$this->_mime_types[63]['mime_type'] = '';
		$this->_mime_types[64]['mime_type'] = '';
		$this->_mime_types[65]['mime_type'] = '';
		$this->_mime_types[66]['mime_type'] = '';
		$this->_mime_types[67]['mime_type'] = '';
		$this->_mime_types[68]['mime_type'] = '';
		$this->_mime_types[69]['mime_type'] = '';
		$this->_mime_types[70]['mime_type'] = '';
		$this->_mime_types[71]['mime_type'] = '';
		$this->_mime_types[72]['mime_type'] = '';
		$this->_mime_types[73]['mime_type'] = '';
		$this->_mime_types[74]['mime_type'] = '';
		$this->_mime_types[75]['mime_type'] = '';
		$this->_mime_types[76]['mime_type'] = '';
		$this->_mime_types[77]['mime_type'] = '';
		$this->_mime_types[78]['mime_type'] = '';
		$this->_mime_types[79]['mime_type'] = '';
		$this->_mime_types[80]['mime_type'] = '';
		$this->_mime_types[81]['mime_type'] = '';
		$this->_mime_types[82]['mime_type'] = '';
		$this->_mime_types[83]['mime_type'] = '';
		$this->_mime_types[84]['mime_type'] = 'image/jpeg';
		$this->_mime_types[85]['mime_type'] = 'image/jpeg';
		$this->_mime_types[86]['mime_type'] = 'image/jpeg';
		$this->_mime_types[87]['mime_type'] = 'image/jpeg';
		$this->_mime_types[88]['mime_type'] = 'image/gif';
	}

	/**
	 * Load known image types
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return 		void
	 * @since 		3.0
	 */
	private function _loadImageTypes() {
		$this->_image_types[0]['mime_type'] = 'image/gif';
		$this->_image_types[1]['mime_type'] = 'image/bmp';
		$this->_image_types[2]['mime_type'] = 'image/png';
		$this->_image_types[3]['mime_type'] = 'image/jpeg';
		$this->_image_types[4]['mime_type'] = 'image/jpeg';
		$this->_image_types[5]['mime_type'] = 'image/gif';
	}

	/**
	 * Convert/Resize an image
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		public
	 * @param 		array	$thumb_file_details	contains all the variables for creating a new image
	 * @return 		mixed	filename of created file if file has been created / false if file has not been created
	 * @since 		3.0
	 */
	public function convertImage($thumb_file_details) {
		// Set all details
		foreach ($thumb_file_details as $type => $value) {
			switch ($type) {
				case 'maxsize':
					if ($value) $this->maxSize = true;
					else $this->maxSize = false;
					break;
				case 'bgred':
					if ($thumb_file_details['bgred'] >= 0 || $thumb_file_details['bgred'] <= 255) $this->bg_red = $thumb_file_details['bgred'];
					else $this->bg_red = 0;
					break;
				case 'bggreen':
					if($thumb_file_details['bggreen'] >= 0 || $thumb_file_details['bggreen'] <= 255) $this->bg_green = $thumb_file_details['bggreen'];
					else $this->bg_green = 0;
					break;
				case 'bgblue':
					if($thumb_file_details['bgblue'] >= 0 || $thumb_file_details['bgblue'] <= 255) $this->bg_blue = $thumb_file_details['bgblue'];
					else $this->bg_blue = 0;
					break;
				default:
					$this->$type = $value;
					break;
			}
		}
		if ($this->newImgCreate()) {
			return $this->file_out;
		}
		else return false;
	}

	/**
	 * Start creating the new image
	 *
	 * @copyright
	 * @author		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return 		boolean	true on success | false on failure
	 * @since 		3.0
	 */
	private function newImgCreate() {
		$csvilog = JRequest::getVar('csvilog');
		// Clear the cache
		clearstatcache();

		switch(strtolower($this->mime_type)) {
			case 'image/gif':
				if( function_exists('imagecreatefromgif') ) {
					$orig_img = @imagecreatefromgif($this->file);
					break;
				}
				else return false;
			case 'image/jpg':
			case 'image/jpeg':
				if (function_exists('imagecreatefromjpeg')) {
					$orig_img = @imagecreatefromjpeg($this->file);
					break;
				}
				else {
					return false;
				}
				break;
			case 'image/png':
				if( function_exists('imagecreatefrompng') ) {
					$orig_img = @imagecreatefrompng($this->file);
					break;
				}
				else return false;
				break;
			default:
				return false;
				break;
		}
		if ($orig_img) {
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_SAVING_NEW_IMAGE'));
			// Save the new image
			$img_resize = $this->NewImgSave($this->NewImgResize($orig_img));
			// Clean up old image
			ImageDestroy($orig_img);
		}
		else {
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_CANNOT_READ_ORIGINAL_IMAGE'));
			$img_resize = false;
		}

		if ($img_resize) return true;
		else return false;
	}

	/**
	 * Resize the image
	 *
	 * Includes function ImageCreateTrueColor and ImageCopyResampled which are available only under GD 2.0.1 or higher !
	 *
	 * @copyright
	 * @author		RolandD
	 * @todo		Fix docbloc
	 * @see
	 * @access 		private
	 * @param 		$orig_img
	 * @return
	 * @since 		3.0
	 */
	private function NewImgResize($orig_img) {
		$orig_size = getimagesize($this->file);

		$maxX = $this->file_out_width;
		$maxY = $this->file_out_height;

		if ($orig_size[0] < $orig_size[1]) {
			$this->file_out_width = $this->file_out_height* ($orig_size[0]/$orig_size[1]);
			$adjustX = ($maxX - $this->file_out_width)/2;
			$adjustY = 0;
		}
		else {
			$this->file_out_height = $this->file_out_width / ($orig_size[0]/$orig_size[1]);
			$adjustX = 0;
			$adjustY = ($maxY - $this->file_out_height)/2;
		}

		while ($this->file_out_width < 1 || $this->file_out_height < 1) {
			$this->file_out_width*= 2;
			$this->file_out_height*= 2;
		}

		// See if we need to create an image at maximum size
		if ($this->maxSize) {
			if (function_exists("imagecreatetruecolor")) $im_out = imagecreatetruecolor($maxX,$maxY);
			else $im_out = imagecreate($maxX,$maxY);

			if ($im_out) {
				// Need to image fill just in case image is transparent, don't always want black background
				$bgfill = imagecolorallocate( $im_out, $this->bg_red, $this->bg_green, $this->bg_blue );

				if (function_exists("imageAntiAlias")) imageAntiAlias($im_out,true);
				imagealphablending($im_out, false);

				if (function_exists("imagesavealpha")) imagesavealpha($im_out,true);

				if (function_exists( "imagecolorallocatealpha")) $transparent = imagecolorallocatealpha($im_out, 255, 255, 255, 127);

				if (function_exists("imagecopyresampled")) ImageCopyResampled($im_out, $orig_img, $adjustX, $adjustY, 0, 0, $this->file_out_width, $this->file_out_height,$orig_size[0], $orig_size[1]);
				else ImageCopyResized($im_out, $orig_img, $adjustX, $adjustY, 0, 0, $this->file_out_width, $this->file_out_height,$orig_size[0], $orig_size[1]);
			}
			else return false;
		}
		else {
			if (function_exists("imagecreatetruecolor")) $im_out = ImageCreateTrueColor($this->file_out_width,$this->file_out_height);
			else $im_out = imagecreate($this->file_out_width,$this->file_out_height);

			if ($im_out) {
				if (function_exists("imageAntiAlias")) imageAntiAlias($im_out,true);
				imagealphablending($im_out, false);

				if (function_exists("imagesavealpha")) imagesavealpha($im_out,true);
				if (function_exists("imagecolorallocatealpha")) $transparent = imagecolorallocatealpha($im_out, 255, 255, 255, 127);

				if (function_exists("imagecopyresampled")) ImageCopyResampled($im_out, $orig_img, 0, 0, 0, 0, $this->file_out_width, $this->file_out_height,$orig_size[0], $orig_size[1]);
				else ImageCopyResized($im_out, $orig_img, 0, 0, 0, 0, $this->file_out_width, $this->file_out_height,$orig_size[0], $orig_size[1]);
			}
			else return false;
		}

		return $im_out;
	}

	/**
	 * Save the new image
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo		Add check if destination folder exists
	 * @todo		Fix docbloc
	 * @see
	 * @access 		private
	 * @param 		$new_img
	 * @return
	 * @since 		3.0
	 */
	private function NewImgSave($new_img) {
		$csvilog = JRequest::getVar('csvilog');
		// Lets see if we need to rename the output file since we know the sizes
		if ($this->rename) {
			$filename = basename($this->file_out, '.'.$this->file_extension);
			$filename_new = $filename.'_'.floor($this->file_out_width).'x'.floor($this->file_out_height).'.'.$this->file_extension;
			$this->file_out = str_ireplace($filename.'.'.$this->file_extension, $filename_new, $this->file_out);
		}
		switch(strtolower($this->file_out_extension)) {
			case "gif":
				if (strtolower(substr($this->file_out,strlen($this->file_out)-4,4)) != ".gif") $this->file_out .= ".gif";
				return @imagegif($new_img, $this->file_out);
				break;
			case "jpg":
				if (strtolower(substr($this->file_out,strlen($this->file_out)-4,4)) != ".jpg") $this->file_out .= ".jpg";
				return @imagejpeg($new_img, $this->file_out, 100);
				break;
			case "jpeg":
				if (strtolower(substr($this->file_out,strlen($this->file_out)-5,5)) != ".jpeg") $this->file_out .= ".jpeg";
				return @imagejpeg($new_img, $this->file_out, 100);
				break;
			case "png":
				if (strtolower(substr($this->file_out,strlen($this->file_out)-4,4)) != ".png") $this->file_out .= ".png";
				return @imagepng($new_img,$this->file_out);
				break;
			default:
				$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_NO_FILE_EXTENSION'));
				return false;
				break;
		}
	}

	/**
	 * Create a thumbnail from the image file on import
	 * A thumbnail will also be created if the thumbnail file already exists
	 *
	 * @copyright
	 * @author		RolandD
	 * @todo 		Add message for non-existing files
	 * @todo 		Move image conversion into a private function
	 * @todo 		Use JFile::getExt to get file extension
	 * @todo 		Generate image names with external images???
	 * @todo		Test image files on FTP
	 * @see
	 * @access 		public
	 * @param 		string	$full_image			name of the image to create a thumbnail from (product_full_image)
	 * @param		string	$thumb_image		name of the thumbnail image
	 * @param		string	$file_location		the location where the images are stored
	 * @param		string	$full_image_output	name of the output full image
	 * @param		string	$thumb_image_output	name of the output full image
	 * @return 		mixed	array of details | false if full image is not found
	 * @since 		3.0
	 */
	public function processImage($full_image, $thumb_image, $file_location, $full_image_output=null, $thumb_image_output=null) {
		// Set up variables
		$template = JRequest::getVar('template');
		$csvilog = JRequest::getVar('csvilog');

		$file_details = array();
		$create_image = false;
		$this->_imagedata = array();

		// Assign the data first
		$this->_imagedata['original']['full_image'] = $full_image;
		$this->_imagedata['original']['thumb_image'] = $thumb_image;
		$this->_imagedata['original']['file_location'] = $file_location;
		$this->_imagedata['original']['full_image_output'] = $full_image_output;
		$this->_imagedata['original']['thumb_image_output'] = $thumb_image_output;
		$this->_imagedata['full_image']['isremote'] = false;
		$this->_imagedata['full_image']['exists'] = false;
		$this->_imagedata['full_image']['isimage'] = false;

		// Load externals
		jimport('joomla.filesystem.folder');
		jimport('joomla.filesystem.file');

		// See if we need to handle a remote file
		if ($this->_isRemote($full_image)) {
			$csvilog->addDebug('Process remote file: '.$full_image);
			$this->_imagedata['full_image']['isremote'] = true;
			if (CsviHelper::fileExistsRemote($full_image)) {
				$this->_imagedata['full_image']['exists'] = true;
				// Check if this is an image or not
				if ($this->isImage($full_image, true)) {
					$this->_imagedata['full_image']['isimage'] = true;
				}
			}
			else {
				$csvilog->addDebug('Remote file does not exist: '.$full_image);
			}
		}
		else if (JFile::exists(JPath::clean($file_location.'/'.$full_image, '/'))) {
			// Clean the file paths
			$full_image = JPath::clean($full_image, '/');
			$thumb_image = JPath::clean($thumb_image, '/');
			$file_location = JPath::clean($file_location, '/');

			$csvilog->addDebug('Process file: '.$file_location.'/'.$full_image);
			$this->_imagedata['full_image']['exists'] = true;
			// Check if this is an image or not
			if ($this->isImage($file_location.'/'.$full_image)) {
				$this->_imagedata['full_image']['isimage'] = true;
			}
		}
		else {
			// File does not exist
			$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_DEBUG_FILE_NOT_FOUND', $file_location.'/'.$full_image));
			return $this->_imagedata;
		}

		if ($this->_imagedata['full_image']['isimage']) {
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_DEBUG_PROCESS_IMAGE'));

			// Clean up the images first
			$this->_cleanupFullImage();
			$this->_cleanupThumbImage();

			// Convert the full image
			if ($template->getValue('convert_type', 'image') != 'none') {
				$this->_convertImage();
			}

			// Create thumbnails
			if ($template->getValue('thumb_create', 'image')) {
				// Sanitize filename
				$this->_imagedata['thumb_image']['output_filename'] = $this->_cleanFilename($this->_imagedata['thumb_image']['output_filename']);
				$this->_createThumbnail();
			}

			// Save the remote images on the server
			if ($template->getValue('save_images_on_server', 'image')) {
				// Sanitize filename
				$this->_imagedata['full_image']['output_name'] = $this->_cleanFilename($this->_imagedata['full_image']['output_name']);
				$from = $this->_checkFile('full', 'original', false, true);
				$to = $this->_checkFile('full', 'output');
				$csvilog->addDebug('Store remote file on server '.$from.' --> '.$to);
				$this->_storeTempImage($from, $to);

				// Check if we need to remove any thumbnails stored
				if ($this->_imagedata['thumb_image']['isremote']) {
					$this->_removeTempImage($this->_imagedata['thumb_image']['folder'].'/'.$this->_imagedata['thumb_image']['name']);
				}
			}
			// Remove temporary file
			else if ($this->_imagedata['full_image']['isremote']) {
				$this->_removeTempImage($this->_imagedata['full_image']['folder'].'/'.$this->_imagedata['full_image']['name']);
			}

			// Check if any images need to be renamed
			if (strtolower($this->_imagedata['full_image']['extension']) != strtolower($this->_imagedata['full_image']['output_extension'])) {
				$this->_convertImage();
			}
			else $this->_renameImage();

			// Check if the full image needs to be resized
			$this->_resizeFullimage();

			// Collect image details
			if (strtolower($template->getValue('import_type')) == 'productfilesimport') $this->_collectImageDetails();
		}
		else {
			if ($this->_imagedata['full_image']['isremote']) $csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_DEBUG_FILE_IS_NOT_IMAGE').' '.$full_image);
			else $csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_DEBUG_FILE_IS_NOT_IMAGE').' '.$file_location.'/'.$full_image);
			// Collect extra info
			$this->_imagedata['full_image']['mime_type'] = $this->findMimeType($file_location.'/'.$full_image);
			// $this->_imagedata['full_image']['output_extension'] = $this->_imagedata['full_image']['extension'];

			// Non image details
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_DEBUG_PROCESS_NON_IMAGE'));
			$this->collectFileDetails();
		}

		// Convert images
		// $this->_imageTypeCheck();

		// Return an array with file details
		return $this->_imagedata;
	}

	/**
	 * Check if a file is a remote file or not
	 *
	 * Remote images can be located on an HTTP location or an FTP location
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see 		processImage()
	 * @access 		private
	 * @param 		$path 	string	the full path to check
	 * @return		bool 	true if file is remote | false if file is not remote
	 * @since 		3.0
	 */
	private function _isRemote($path) {
		if (substr(strtolower($path), 0, 4) == 'http') return true;
		else if (substr(strtolower($path), 0, 3) == 'ftp') return true;
		else return false;
	}

	/**
	 * Clean up the full image
	 *
	 * Clean up the image from any incorrect paths
	 *
	 * Minimum requirement is PHP 5.2.0
	 *
	 * [full_image] => Array
     *  (
     *      [isremote] => 1
     *      [exists] => 1
     *      [isimage] => 1
     *      [name] => R05-01 -- R05-01 (700).jpg
     *      [filename] => R05-01 -- R05-01 (700)
     *      [extension] => jpg
     *      [folder] => http://csvi3
     *      [output_name] => R05-01 -- R05-01 (700).jpg
     *      [output_filename] => R05-01 -- R05-01 (700)
     *      [output_extension] => jpg
     *      [output_folder] => http://csvi3
     *      [mime_type] => image/jpeg
     *  )
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo		handle image paths included in the image name
	 * @todo		not delete the old image, it might be referenced by another product
	 * @see 		http://www.php.net/pathinfo
	 * @access 		private
	 * @param
	 * @return
	 * @since 		3.0
	 */
	private function _cleanupFullImage() {
		$csvilog = JRequest::getVar('csvilog');
		$template = JRequest::getVar('template');

		// Get some file info
		$pathinfo_full_image = pathinfo($this->_imagedata['original']['full_image']);

		// Incoming
		$this->_imagedata['full_image']['name'] = $pathinfo_full_image['basename'];
		$this->_imagedata['full_image']['filename'] = $pathinfo_full_image['filename'];
		$this->_imagedata['full_image']['extension'] = (array_key_exists('extension', $pathinfo_full_image)) ? $pathinfo_full_image['extension'] : '';
		$this->_imagedata['full_image']['folder'] = (array_key_exists('dirname', $pathinfo_full_image)) ? (($pathinfo_full_image['dirname'] == '.') ? '' : $pathinfo_full_image['dirname']) : '';

		// Outgoing
		if (!is_null($this->_imagedata['original']['full_image_output'])) {
			// Get some file info
			$pathinfo_full_image_output = pathinfo($this->_imagedata['original']['full_image_output']);

			$this->_imagedata['full_image']['output_name'] = $pathinfo_full_image_output['basename'];
			$this->_imagedata['full_image']['output_filename'] = $pathinfo_full_image_output['filename'];
			$this->_imagedata['full_image']['output_extension'] = (array_key_exists('extension', $pathinfo_full_image_output)) ? $pathinfo_full_image_output['extension'] : '';
		}
		else {
			$this->_imagedata['full_image']['output_name'] = $this->_imagedata['full_image']['name'];
			$this->_imagedata['full_image']['output_filename'] = $this->_imagedata['full_image']['filename'];
			$this->_imagedata['full_image']['output_extension'] =$this->_imagedata['full_image']['extension'];
		}

		// Set the output folder
		if (empty($this->_imagedata['full_image']['folder']) || $this->_imagedata['full_image']['isremote']) $this->_imagedata['full_image']['output_folder'] = $this->_imagedata['full_image']['folder'];
		else if (!empty($this->_imagedata['full_image']['folder'])) $this->_imagedata['full_image']['output_folder'] = JPath::clean($this->_imagedata['full_image']['folder'], '/');

		if ($this->_imagedata['full_image']['isremote'] && $template->getValue('save_images_on_server', 'image')) {
			// Collect remote file information
			$this->_imagedata['full_remote']['filename'] = mt_rand();
			$this->_imagedata['full_remote']['folder'] = CSVIPATH_TMP;
			// Build full filenames
			if (!strstr($this->_imagedata['full_image']['name'], '?')) $remote_image = $this->_imagedata['full_image']['output_folder'].'/'.rawurlencode($this->_imagedata['full_image']['name']);
			else $remote_image = $this->_imagedata['full_image']['output_folder'].'/'.$this->_imagedata['full_image']['name'];
			$local_image_name = $this->_cleanFilename($this->_imagedata['full_remote']['filename'].'.'.$this->_imagedata['full_image']['extension']);
			$local_image = $this->_imagedata['full_remote']['folder'].'/'.$local_image_name;
			// Store the remote image
			if ($this->_storeRemoteImage($remote_image, $local_image)) {
				$csvilog->addDebug('Remote file stored: '.$remote_image.' --> '.$local_image);
				// Update full image information
				$this->_imagedata['full_image']['name'] = $local_image_name;
				$this->_imagedata['full_image']['filename'] = $this->_imagedata['full_remote']['filename'];
				$this->_imagedata['full_image']['folder'] = $this->_imagedata['full_remote']['folder'];
				$this->_imagedata['full_image']['output_folder'] = '';

				// Get the mime type
				$mime_type = $this->findMimeType($local_image);
			}
			else {
				$csvilog->AddStats('nofiles', JText::sprintf('COM_CSVIVIRTUEMART_REMOTE_FILE_NOT_FOUND', $remote_image));
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_REMOTE_FILE_NOT_FOUND', $remote_image));
			}
		}
		else if ($this->_imagedata['full_image']['isremote']) {
			$mime_type = $this->findMimeType($this->_imagedata['original']['full_image'], true);
		}
		else if (!$this->_imagedata['full_image']['isremote']) {
			$mime_type = $this->findMimeType($this->_imagedata['original']['file_location'].'/'.$this->_imagedata['original']['full_image']);
		}

		// Set the mime type
		$csvilog->addDebug('Mime type found: '.$mime_type);
		$this->_imagedata['full_image']['mime_type'] = $mime_type;

		// Validate extension against mime type
		$type = '';
		$ext = '';
		list($type, $ext) = explode('/', $mime_type);
		if ($ext == 'jpeg') $ext = 'jpg';
		if ($ext != strtolower($this->_imagedata['full_image']['extension'])) {
			// Build the from name
			$from = $this->_checkFile('full', 'original', false);

			// Fix up the new names
			if (!in_array(strtolower($this->_imagedata['full_image']['extension']), $this->_supported_images)) {
				// Add pseudo extension
				$to = $from.'.'.$ext;
				// Set the new extensions
				$this->_imagedata['full_image']['name'] .= '.'.$ext;
				// Set the new output name
				$this->_imagedata['full_image']['output_name'] .= '.'.$ext;
			}
			else {
				$this->_imagedata['full_image']['name'] = $this->_imagedata['full_image']['filename'].'.'.$ext;
				// Set the new output name
				$this->_imagedata['full_image']['output_name'] = $this->_imagedata['full_image']['output_filename'].'.'.$ext;
				// Build the to name
				$to = JFile::stripExt($from).'.'.$ext;
			}

			$csvilog->addDebug('Renaming full image because bad extension: '.$from.' --> '.$to);

			// Rename the file
			if (JFile::exists($from)) {
				if (!JFile::move($from, $to)) return false;
				else {
					$this->_imagedata['full_image']['extension'] = $ext;
					$this->_imagedata['full_image']['output_extension'] = $ext;
				}
			}
		}

		// Check for a valid extenion
		if (!empty($this->_imagedata['full_image']['extension'])) {
			$csvilog->addDebug('File has extension: '.$this->_imagedata['full_image']['extension']);
			// Check if the full name is an image name
			if (!in_array(strtolower($this->_imagedata['full_image']['extension']), $this->_supported_images)) {
				// Full name is not a regular image name
				$csvilog->addDebug('Full name is not a regular image name');
				if ($type == 'image') {
					$this->_imagedata['full_image']['extension'] = $ext;
					$this->_imagedata['full_image']['output_extension'] = $ext;
				}
			}
		}
		else {
			// Full image name has no extension
			$csvilog->addDebug('Full name is not a regular image name');
			if ($type == 'image') {
				$this->_imagedata['full_image']['extension'] = $ext;
				$this->_imagedata['full_image']['output_extension'] = $ext;
			}
		}

		// Set a new extension if the image needs to be converted
		$convert_type = $template->getValue('convert_type', 'image');
		if ($convert_type != 'none' && $convert_type != $this->_imagedata['full_image']['extension']) {
			$this->_imagedata['full_image']['output_name'] = $this->_imagedata['full_image']['filename'].'.'.$convert_type;
			$this->_imagedata['full_image']['output_extension'] = $convert_type;
		}

		// Set the file case
		$this->_imagedata['full_image']['output_name'] = $this->_setCase($this->_imagedata['full_image']['output_name']);
		$this->_imagedata['full_image']['output_filename'] = $this->_setCase($this->_imagedata['full_image']['output_filename']);
		$this->_imagedata['full_image']['output_extension'] = $this->_setCase($this->_imagedata['full_image']['output_extension']);

		// Add some debug info
		$csvilog->addDebug('Full name original: '.$this->_imagedata['original']['full_image']);
		$csvilog->addDebug('Full name target: '.$this->_imagedata['full_image']['output_name']);
		$csvilog->addDebug('Local folder: '.$this->_imagedata['full_image']['output_folder']);
	}

	/**
	 * Clean up the thumbnail image
	 *
	 * Clean up the image from any incorrect paths
	 *
	 * Minimum requirement is PHP 5.2.0
	 *
	 *[thumb_image] => Array
     *  (
     *      [name] => R05-01 -- R05-01 (700).jpg
     *      [filename] => R05-01 -- R05-01 (700)
     *      [extension] => jpg
     *      [folder] =>
     *      [output_name] => R05-01 -- R05-01 (700).jpg
     *      [output_filename] => R05-01 -- R05-01 (700)
     *      [output_extension] => jpg
     *      [output_folder] => resized
     *      [output_rename] => 0
     *      [isremote] =>
     *  )
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo		handle image paths included in the image name
	 * @see 		http://www.php.net/pathinfo
	 * @access 		private
	 * @param
	 * @return
	 * @since 		3.0
	 */
	private function _cleanupThumbImage() {
		$csvilog = JRequest::getVar('csvilog');
		$template = JRequest::getVar('template');

		// Check if the thumbnail image is empty
		if (!empty($this->_imagedata['original']['thumb_image'])) {
			// Get some file info
			$pathinfo_thumb_image = pathinfo($this->_imagedata['original']['thumb_image']);
			// Incoming
			$this->_imagedata['thumb_image']['name'] = $pathinfo_thumb_image['basename'];
			$this->_imagedata['thumb_image']['filename'] = $pathinfo_thumb_image['filename'];
			$this->_imagedata['thumb_image']['extension'] = (array_key_exists('extension', $pathinfo_thumb_image)) ? $pathinfo_thumb_image['extension'] : '';
			$this->_imagedata['thumb_image']['folder'] = (array_key_exists('dirname', $pathinfo_thumb_image)) ? (($pathinfo_thumb_image['dirname'] == '.') ? '' : $pathinfo_thumb_image['dirname']) : '';

			// Outgoing
			if (!is_null($this->_imagedata['original']['thumb_image_output'])) {
				// Get some file info
				$pathinfo_thumb_image_output = pathinfo($this->_imagedata['original']['thumb_image_output']);

				$this->_imagedata['thumb_image']['output_name'] = $pathinfo_thumb_image_output['basename'];
				$this->_imagedata['thumb_image']['output_filename'] = $pathinfo_thumb_image_output['filename'];
				$this->_imagedata['thumb_image']['output_extension'] = (array_key_exists('extension', $pathinfo_thumb_image_output)) ? $pathinfo_thumb_image_output['extension'] : '';
			}
			else {
				$this->_imagedata['thumb_image']['output_name'] = $this->_imagedata['thumb_image']['name'];
				$this->_imagedata['thumb_image']['output_filename'] = $this->_imagedata['thumb_image']['filename'];
				$this->_imagedata['thumb_image']['output_extension'] = $this->_imagedata['thumb_image']['extension'];
			}

			// Set the output folder
			if (empty($this->_imagedata['thumb_image']['folder'])) $this->_imagedata['thumb_image']['output_folder'] = '';
			else {
				if (!$this->_isRemote($this->_imagedata['thumb_image']['folder'])) {
					$this->_imagedata['thumb_image']['output_folder'] = JPath::clean($this->_imagedata['thumb_image']['folder'], '/');
				}
				else $this->_imagedata['thumb_image']['output_folder'] = $this->_imagedata['thumb_image']['folder'];
			}

			// Check for a product files import
			if (strtolower($template->getValue('import_type')) == 'productfilesimport') {
				$this->_imagedata['thumb_image']['output_folder'] .= '/resized';
				$this->_imagedata['thumb_image']['output_rename'] = 1;
			}
			else $this->_imagedata['thumb_image']['output_rename'] = 0;

			// Make sure the thumbnail is the same file type as the full image
			if ($template->getValue('thumb_check_filetype', 'image') && $this->_imagedata['thumb_image']['output_extension'] != $this->_imagedata['full_image']['output_extension']) {
				$this->_imagedata['thumb_image']['output_extension'] = $this->_imagedata['full_image']['output_extension'];
			}

			// See if we have a remote image
			$this->_imagedata['thumb_image']['isremote'] = $this->_isRemote($this->_imagedata['original']['thumb_image']);

			// Handle remote thumbnails
			if ($this->_imagedata['thumb_image']['isremote'] && $template->getValue('save_images_on_server', 'image')) {
				// Collect remote file information
				$this->_imagedata['thumb_remote']['filename'] = mt_rand();
				$this->_imagedata['thumb_remote']['folder'] = CSVIPATH_TMP;
				// Build full filenames
				$remote_image = $this->_imagedata['original']['thumb_image'];
				$local_image = $this->_imagedata['thumb_remote']['folder'].'/'.$this->_imagedata['thumb_remote']['filename'].'.'.$this->_imagedata['thumb_image']['extension'];
				// Store the remote image
				if ($this->_storeRemoteImage($remote_image, $local_image)) {
					$csvilog->addDebug('Remote thumb file stored: '.$remote_image.' --> '.$local_image);
					// Update thumb image information
					$this->_imagedata['thumb_image']['name'] = $this->_imagedata['thumb_remote']['filename'].'.'.$this->_imagedata['thumb_image']['extension'];
					$this->_imagedata['thumb_image']['filename'] = $this->_imagedata['thumb_remote']['filename'];
					$this->_imagedata['thumb_image']['folder'] = $this->_imagedata['thumb_remote']['folder'];
					$this->_imagedata['thumb_image']['output_folder'] = 'resized';
				}
				else {
					$csvilog->AddStats('nofiles', JText::sprintf('COM_CSVIVIRTUEMART_REMOTE_FILE_NOT_FOUND', $remote_image));
					$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_REMOTE_FILE_NOT_FOUND', $remote_image));
				}
			}
			else if ($template->getValue('thumb_create', 'image')) $this->_imagedata['thumb_image']['output_folder'] = 'resized';

			// Thumbnail extension
			if (strtolower($template->getValue('thumb_extension', 'image')) != 'none') $this->_imagedata['thumb_image']['output_extension'] = $template->getValue('thumb_extension', 'image');
		}
		else {
			// Thumbnail is empty
			// Let's see if we have an output name
			if (!empty($this->_imagedata['original']['thumb_image_output'])) {
				$pathinfo_thumb_image = pathinfo($this->_imagedata['original']['thumb_image_output']);
				$this->_imagedata['thumb_image']['output_name'] = $pathinfo_thumb_image['basename'];
				$this->_imagedata['thumb_image']['output_filename'] = $pathinfo_thumb_image['filename'];
				$this->_imagedata['thumb_image']['output_extension'] = (array_key_exists('extension', $pathinfo_thumb_image)) ? $pathinfo_thumb_image['extension'] : '';
				$this->_imagedata['thumb_image']['output_folder'] = (array_key_exists('dirname', $pathinfo_thumb_image)) ? (($pathinfo_thumb_image['dirname'] == '.') ? '' : $pathinfo_thumb_image['dirname']) : '';
			}
			else {
				$this->_imagedata['thumb_image']['output_name'] = '';
				$this->_imagedata['thumb_image']['output_filename'] = '';
				$this->_imagedata['thumb_image']['output_extension'] = '';
				$this->_imagedata['thumb_image']['output_folder'] = '';
			}
			$this->_imagedata['thumb_image']['name'] = '';
			$this->_imagedata['thumb_image']['output_rename'] = 0;
			$this->_imagedata['thumb_image']['isremote'] = false;
		}

		// Set the file case
		$this->_imagedata['thumb_image']['output_name'] = $this->_setCase($this->_imagedata['thumb_image']['output_name']);
		$this->_imagedata['thumb_image']['output_filename'] = $this->_setCase($this->_imagedata['thumb_image']['output_filename']);
		$this->_imagedata['thumb_image']['output_extension'] = $this->_setCase($this->_imagedata['thumb_image']['output_extension']);

		// Add some debug info
		$csvilog->addDebug('Thumb name original: '.$this->_imagedata['original']['thumb_image']);
		$csvilog->addDebug('Thumb name target: '.$this->_imagedata['thumb_image']['output_name']);
		$csvilog->addDebug('Thumb folder: '.$this->_imagedata['thumb_image']['output_folder']);
	}

	/**
	 * Store a remote image on the local server
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param 		$remote_image 	string	the url of the remote image
	 * @param 		$local_image	string	the full path and file name of the image to store
	 * @return 		bool true if remote file was locally written | false if remote file was not locally written
	 * @since
	 */
	private function _storeRemoteImage($remote_image, $local_image) {
		return JFile::write($local_image, JFile::read($remote_image));
	}

	/**
	 * Convert the full image to another type
	 *
	 * @copyright
	 * @author		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return
	 * @since 		3.0
	 */
	private function _convertImage() {
		$csvilog = JRequest::getVar('csvilog');
		$template = JRequest::getVar('template');

		// Let's see if the extensions are the same
		if (strtolower($this->_imagedata['full_image']['extension']) == strtolower($this->_imagedata['full_image']['output_extension'])) {
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_IMAGE_NOT_CONVERTED_EXT_SAME'));
		}
		else if ($this->_isRemote($this->_checkFile('full', 'output'))) {
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_IMAGE_NOT_CONVERTED_REMOTE_FILE'));
		}
		else {
			// Create file to check
			$checkfile = $this->_checkFile('full', 'original');

			// Collect the image details
			$file_details = array();
			$file_details['file'] = $checkfile;
			$file_details['file_extension'] = $this->_imagedata['full_image']['extension'];
			$file_details['rename'] = 0;
			$file_details['file_out'] = $this->_checkFile('full', 'output');
			$file_details['maxsize'] = 0;
			$file_details['bgred'] = 255;
			$file_details['bggreen'] = 255;
			$file_details['bgblue'] = 255;
			$new_sizes = getimagesize($checkfile);
			$file_details['file_out_width'] = $new_sizes[0];
			$file_details['file_out_height'] = $new_sizes[1];
			$file_details['file_out_extension'] = $this->_imagedata['full_image']['output_extension'];
			$file_details['mime_type'] = $this->_imagedata['full_image']['mime_type'];

			// We need to resize the image and Save the new one (all done in the constructor)
			$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_DEBUG_CONVERT_IMAGE', $file_details['file'], $file_details['file_out']));
			$new_img = $this->convertImage($file_details);

			if ($new_img) {
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_IMAGE_CONVERTED', $file_details['file']));
				if (JFile::exists($file_details['file'])) JFile::delete($file_details['file']);
				// We have a new name, so refresh the info
				$pathinfo_full_image = pathinfo(basename($new_img));
				$this->_imagedata['full_image']['output_name'] = $pathinfo_full_image['basename'];
				$this->_imagedata['full_image']['output_filename'] = $pathinfo_full_image['filename'];
				$this->_imagedata['full_image']['output_extension'] = $pathinfo_full_image['extension'];
				$this->_imagedata['full_image']['mime_type'] = $this->findMimeType($file_details['file_out']);
				if ($this->_imagedata['full_image']['isremote']) {
					$this->_imagedata['full_image']['name'] = $pathinfo_full_image['basename'];
					$this->_imagedata['full_image']['filename'] = $pathinfo_full_image['filename'];
					$this->_imagedata['full_image']['extension'] = $pathinfo_full_image['extension'];
					$this->_imagedata['full_image']['isremote'] = false;
					$this->_imagedata['full_image']['folder'] = '';
				}
				return true;
			}
			else {
				$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_IMAGE_NOT_CONVERTED'));
				return false;
			}
		}
	}

	/**
	* Create a thumbnail image
	*
	* @author RolandD
	* @access private
	* @param
	* @return
	*/
	private function _createThumbnail() {
		$template = JRequest::getVar('template');
		$csvilog = JRequest::getVar('csvilog');

		// Check if thumbsize is greater than 0
		if ($template->getValue('thumb_width', 'image') >= 1 && $template->getValue('thumb_height', 'image') >= 1) {
			// Check if the image folders exists
			$thumb_folder = $this->_imagedata['original']['file_location'].'/'.$this->_imagedata['thumb_image']['output_folder'];
			if (!JFolder::exists($thumb_folder)) {
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_CREATE_THUMB_FOLDER', $thumb_folder));
				JFolder::create($thumb_folder);
			}

			// Create file to check
			$checkfile = $this->_checkFile('full', 'original');

			// Check if the original file exists
			$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_CHECK_ORIGINAL', $checkfile));
			if (JFile::exists($checkfile) || $this->_imagedata['full_image']['isremote']) {
				// Collect all thumbnail details
				$thumb_file_details = array();
				$thumb_file_details['file'] = $checkfile;
				$thumb_file_details['file_extension'] = $this->_imagedata['full_image']['extension'];
				$thumb_file_details['rename'] = $this->_imagedata['thumb_image']['output_rename'];
				$thumb_file_details['file_out'] = $this->_checkFile('thumb', 'output', true);
				$thumb_file_details['maxsize'] = 0;
				$thumb_file_details['bgred'] = 255;
				$thumb_file_details['bggreen'] = 255;
				$thumb_file_details['bgblue'] = 255;
				$thumb_file_details['file_out_width'] = $template->getValue('thumb_width', 'image');
				$thumb_file_details['file_out_height'] = $template->getValue('thumb_height', 'image');
				$thumb_file_details['file_out_extension'] = $this->_imagedata['thumb_image']['output_extension'];
				$thumb_file_details['mime_type'] = $this->_imagedata['full_image']['mime_type'];

				// We need to resize the image and Save the new one (all done in the constructor)
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_DEBUG_CREATING_A_THUMBNAIL', $checkfile, $thumb_file_details['file_out']));
				$new_img = $this->convertImage($thumb_file_details);

				// Check if an image was created
				if ($new_img) {
					// Get the details of the thumb image
					if (JFile::exists($new_img)) {
						$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_DEBUG_THUMB_CREATED'));
						$new_details = pathinfo($new_img);
						$this->_imagedata['thumb_image']['output_name'] = $new_details['basename'];
						$this->_imagedata['thumb_image']['output_filename'] = $new_details['filename'];
						$this->_imagedata['thumb_image']['output_extension'] = $new_details['extension'];
						$file_details_thumb = getimagesize($new_img);
						if ($file_details_thumb) {
							$this->_imagedata['thumb_image']['output_width'] = $file_details_thumb[0];
							$this->_imagedata['thumb_image']['output_height'] = $file_details_thumb[1];
						}
						else {
							$this->_imagedata['thumb_image']['output_width'] = 0;
							$this->_imagedata['thumb_image']['output_height'] = 0;
						}
					}
					else {
						$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_THUMBNAIL_NOT_CREATED'));
						return false;
					}
				}
				else {
					$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_THUMBNAIL_NOT_CREATED'));
					return false;
				}
			}
			else {
				$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_FILE_DOES_NOT_EXIST_NOTHING_TO_DO').' '.$this->_imagedata['full_image']['folder'].'/'.$this->_imagedata['full_image']['name']);
				$csvilog->AddStats('nofiles', JText::_('COM_CSVIVIRTUEMART_FILE_DOES_NOT_EXIST_NOTHING_TO_DO').' '.$this->_imagedata['full_image']['folder'].'/'.$this->_imagedata['full_image']['name']);
				return false;
			}
		}
		else {
			$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_THUMBNAIL_SIZE_TOO_SMALL'));
			$csvilog->AddStats('incorrect', JText::_('COM_CSVIVIRTUEMART_THUMBNAIL_SIZE_TOO_SMALL'));
			return false;
		}
	}

	/**
	 * Rename image
	 *
	 * Rename an image, any existing file will be deleted
	 *
	 * @copyright
	 * @author		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return
	 * @since 		3.0
	 */
	private function _renameImage() {
		$csvilog = JRequest::getVar('csvilog');
		$template = JRequest::getVar('template');
		// Full image
		if (!$this->_imagedata['full_image']['isremote']
			&& $template->getValue('auto_generate_image_name', 'image')
			&& ($this->_imagedata['full_image']['name'] != $this->_imagedata['full_image']['output_name'])
			&& $template->getValue('convert_type', 'image') == 'none') {
			$from = $this->_checkFile('full', 'original');
			if (JFile::exists($from)) {
				$to = $this->_checkFile('full', 'output');
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_RENAME_FULL_FILE', $from, $to));
				// Delete any existing target images
				if (JFile::exists($to)) JFile::delete($to);
				// Copy or move the image to its final location
				if ($template->getValue('keep_original', 'image')) JFile::copy($from, $to);
				else JFile::move($from, $to);
			}
			else {
				$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_RENAME_FULL_FILE_NOT_FOUND', $from));
			}
		}

		// Thumb image
		if (!$this->_imagedata['thumb_image']['isremote']
			&& $template->getValue('auto_generate_image_name', 'image')
			&& ($this->_imagedata['thumb_image']['name'] != $this->_imagedata['thumb_image']['output_name'])
			&& !$template->getValue('thumb_create', 'image')
		) {
			$from = $this->_checkFile('thumb', 'original');
			if (JFile::exists($from)) {
				$to = $this->_checkFile('thumb', 'output');
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_RENAME_THUMB_FILE', $from, $to));
				// Delete any existing target images
				if (JFile::exists($to)) JFile::delete($to);
				// Copy or move the image to its final location
				if ($template->getValue('keep_original', 'image')) JFile::copy($from, $to);
				else JFile::move($from, $to);
			}
			else if (!empty($this->imagedata['thumb_image']['original_file'])) {
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_RENAME_THUMB_FILE_NOT_FOUND', $from));
			}
		}
	}

	/**
	* Remove temporary image
	*
	* @author RolandD
	* @access private
	* @var string $image the full path and name of image to remove
	* @return bool true|false
	*/
	private function _removeTempImage($image) {
		if (JFile::exists($image)) return JFile::delete($image);
		else return false;
	}

	/**
	* Store the temporary image
	*
	* @author RolandD
	* @access private
	* @var string $old_image the full path and name of image to move
	* @var string $new_image the full path and name of image to store
	* @return bool true|false
	*/
	private function _storeTempImage($old_image, $new_image) {
		if (JFile::exists($new_image)) JFile::delete($new_image);
		return JFile::move($old_image, $new_image);
	}

	/**
	* Check if we need to convert the final image
	*
	* @author RolandD
	* @since 2.3.7
	* @access private
	* @return void
	*/
	private function _imageTypeCheck() {
		$csvilog = JRequest::getVar('csvilog');
		$template = JRequest::getVar('template');

		/* Create array of checks */
		$checks = array();
		$checks[] = 'full_image';
		$checks[] = 'thumb_image';

		/* Check the full image */
		foreach ($checks as $check) {
			if ($this->_imagedata[$check]['extension'] != $this->_imagedata[$check]['output_extension']) {
				if ($check == 'full_image') $original_file = $this->_checkFile('full', 'output');
				else if ($check == 'thumb_image') $original_file = $this->_checkFile('thumb', 'output');
				$mime_type = $this->findMimeType($original_file);
				if (JFile::exists($original_file) && !stristr($mime_type, $this->_imagedata[$check]['output_extension'])) {
					$file_details = array();
					$file_details['file'] = $original_file;
					if ($check == 'full_image') $file_details['file_out'] = $this->_checkFile('full', 'output');
					else if ($check == 'thumb_image') $file_details['file_out'] = $this->_checkFile('thumb', 'output');
					if ($file_details['file'] != $file_details['file_out']) {
						$file_details['file_extension'] = $this->_imagedata[$check]['extension'];
						$file_details['rename'] = 0;
						$file_details['maxsize'] = 0;
						$file_details['bgred'] = 255;
						$file_details['bggreen'] = 255;
						$file_details['bgblue'] = 255;
						$new_sizes = getimagesize($original_file);
						$file_details['file_out_width'] = $new_sizes[0];
						$file_details['file_out_height'] = $new_sizes[1];
						$file_details['file_out_extension'] = $this->_imagedata[$check]['output_extension'];
						$file_details['mime_type'] = $mime_type;

						/* We need to resize the image and Save the new one (all done in the constructor) */
						$csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_DEBUG_CONVERT_TYPE_CHECK', $file_details['file'], $file_details['file_out']));
						$new_img = $this->convertImage($file_details);

						if ($new_img) $csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_'.strtoupper($check).'_CONVERTED'));
						else $csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_'.strtoupper($check).'_NOT_CONVERTED'));
					}
				}
			}
		}
	}

	/**
	 * Collect image details for the product files import
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return
	 * @since 		2.3.8
	 */
	private function _collectImageDetails() {
		$csvilog = JRequest::getVar('csvilog');
		$template = JRequest::getVar('template');
		// Create the full image path
		if (!$this->_imagedata['full_image']['isremote']) $full_path = JPath::clean($this->_imagedata['original']['file_location'].'/'.$this->_imagedata['full_image']['output_folder'].'/'.$this->_imagedata['full_image']['output_name'], '/');
		else $full_path = $this->_imagedata['full_image']['output_folder'].'/'.$this->_imagedata['full_image']['output_name'];
		$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_DEBUG_COLLECT_FILE_DETAILS', $full_path));

		// Details
		$full_details = getimagesize($full_path);
		if ($full_details) {
			$this->_imagedata['full_image']['image_height'] = $full_details[0];
			$this->_imagedata['full_image']['image_width'] = $full_details[1];
			$this->_imagedata['full_image']['mime_type'] = $full_details['mime'];
		}
		else {
			$this->_imagedata['full_image']['image_height'] = '0';
			$this->_imagedata['full_image']['image_width'] = '0';
			$this->_imagedata['full_image']['mime_type'] = '';
		}

		// Create the thumb image path
		if (!$this->_imagedata['thumb_image']['isremote']) $full_path = JPath::clean($this->_imagedata['original']['file_location'].'/'.$this->_imagedata['thumb_image']['output_folder'].'/'.$this->_imagedata['thumb_image']['output_name'], '/');
		else $full_path = $this->_imagedata['full_image']['output_folder'].'/'.$this->_imagedata['full_image']['output_name'];
		// Details
		if (JFile::exists($full_path)) $full_details = getimagesize($full_path);
		else $full_details = false;
		if ($full_details) {
			$this->_imagedata['thumb_image']['image_height'] = $full_details[0];
			$this->_imagedata['thumb_image']['image_width'] = $full_details[1];
			$this->_imagedata['thumb_image']['mime_type'] = $full_details['mime'];
			$this->_imagedata['thumb_image']['isimage'] = ($full_details) ? '1' : '0';
		}
		else {
			$this->_imagedata['thumb_image']['image_height'] = '0';
			$this->_imagedata['thumb_image']['image_width'] = '0';
			$this->_imagedata['thumb_image']['mime_type'] = '';
			$this->_imagedata['thumb_image']['isimage'] = '0';
		}
	}

	/**
	 * Generates the full path to the full image
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param 		string 	$type		is the type of the file, full or thumb
	 * @param 		string 	$location 	is the destination of the file, original or output
	 * @param 		bool 	$extension 	true the file extension is taken from the imagedata array | false the file extension of the image name is used
	 * @return 		string	the full path of the full image file name
	 * @since 		2.3.14
	 */
	 private function _checkFile($type, $location, $extension=false) {
	 	 $type = strtolower($type);
	 	 switch ($location) {
			 case 'original':
			 	if ($this->_imagedata['full_image']['isremote']) {
					return $this->_imagedata[$type.'_image']['folder'].'/'.$this->_imagedata[$type.'_image']['name'];
			 	}
			 	else {
					if (empty($this->_imagedata[$type.'_image']['folder'])) return $this->_imagedata['original']['file_location'].'/'.$this->_imagedata[$type.'_image']['name'];
					else return $this->_imagedata['original']['file_location'].'/'.$this->_imagedata[$type.'_image']['folder'].'/'.$this->_imagedata[$type.'_image']['name'];
				}
				break;
			case 'output':
				if ($extension) {
					return JPath::clean($this->_imagedata['original']['file_location'].'/'.$this->_imagedata[$type.'_image']['output_folder'].'/'.$this->_imagedata[$type.'_image']['output_filename'].'.'.$this->_imagedata[$type.'_image']['output_extension'], '/');
				}
				else {
					return JPath::clean($this->_imagedata['original']['file_location'].'/'.$this->_imagedata[$type.'_image']['output_folder'].'/'.$this->_imagedata[$type.'_image']['output_name'], '/');
				}
				break;
		 }
	 }

	/**
	 * Collect file details for non-image files
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		public
	 * @param
	 * @return
	 * @since 		2.3.10
	 */
	public function collectFileDetails() {
		$template = JRequest::getVar('template');
		$this->_imagedata['full_image']['mime_type'] = $this->findMimeType($template->getValue('file_location_media', 'path').'/'.$this->_imagedata['original']['full_image']);
		$this->_imagedata['full_image']['isimage'] = 0;
		$this->_imagedata['full_image']['output_extension'] = JFile::getExt($this->_imagedata['original']['full_image']);
		$dirname = dirname($this->_imagedata['original']['full_image']);
		if (!$this->_imagedata['full_image']['isremote']) {
			if ($dirname !== '.') $this->_imagedata['full_image']['output_folder'] = '/'.dirname($this->_imagedata['original']['full_image']);
			else $this->_imagedata['full_image']['output_folder'] = '/';
		}
		else {
			$this->_imagedata['full_image']['output_folder'] = dirname($this->_imagedata['original']['full_image']);
		}
		$this->_imagedata['full_image']['output_name'] =  basename($this->_imagedata['original']['full_image']);
	}

	/**
	 * Clean filename
	 *
	 * Cleans up a filename and replaces non-supported characters with an underscore
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param 		$value	string	the value to clean
	 * @return 		string	the cleaned up value
	 * @since 		3.0
	 */
	private function _cleanFilename($value) {
		return (string) preg_replace('/[^A-Z0-9_\.-\s]/i', '_', $value);
	}

	/**
	 * Change the case of any given string
	 *
	 * @copyright
	 * @author 		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param 		string	$name	the string to be case changed
	 * @return 		string	the case changed string
	 * @since 		3.0
	 */
	private function _setCase($name) {
		$template = JRequest::getVar('template');
		// Set the case if needed
		switch ($template->getValue('change_case', 'image')) {
			case 'lcase':
				return strtolower($name);
				break;
			case 'ucase':
				return strtoupper($name);
				break;
			case 'ucfirst':
				return ucfirst($name);
				break;
			case 'ucwords':
				return ucwords($name);
				break;
			default:
				return $name;
				break;
		}
	}

	/**
	 * Resize a large image
	 *
	 * @copyright
	 * @author		RolandD
	 * @todo
	 * @see
	 * @access 		private
	 * @param
	 * @return
	 * @since 		3.0
	 */
	private function _resizeFullImage() {
		$template = JRequest::getVar('template');
		$csvilog = JRequest::getVar('csvilog');

		// Check if we need to resize the full image
		if ($template->getValue('full_resize', 'image')) {
			// Get the current size
			$checkfile = $this->_checkFile('full', 'output');
			$cur_size = getimagesize($checkfile);
			if ($cur_size[0] > $template->getValue('full_width', 'image') || $cur_size[1] > $template->getValue('full_height', 'image')) {
				// Resize the image
				$file_details = array();
				$file_details['file'] = $checkfile;
				$file_details['file_extension'] = $this->_imagedata['full_image']['output_extension'];
				$file_details['rename'] = 0;
				$file_details['file_out'] = $checkfile;
				$file_details['maxsize'] = 0;
				$file_details['bgred'] = 255;
				$file_details['bggreen'] = 255;
				$file_details['bgblue'] = 255;
				$file_details['file_out_width'] = $template->getValue('full_width', 'image');
				$file_details['file_out_height'] = $template->getValue('full_height', 'image');
				$file_details['file_out_extension'] = $this->_imagedata['full_image']['output_extension'];
				$file_details['mime_type'] = $this->_imagedata['full_image']['mime_type'];

				// We need to resize the image and Save the new one (all done in the constructor)
				$csvilog->addDebug(JText::sprintf('COM_CSVIVIRTUEMART_DEBUG_RESIZE_IMAGE', $file_details['file'], $cur_size[1].'x'.$cur_size[0], $template->getValue('full_height', 'image').'x'.$template->getValue('full_width', 'image')));
				$new_img = $this->convertImage($file_details);

				if ($new_img) $csvilog->addDebug(JText::_('COM_CSVIVIRTUEMART_FULL_IMAGE_RESIZED'));
			}
		}
	}
}
?>